<?php

namespace App\Services;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ProductVariantService
{
    public function handle(Request $request, $product)
    {
        $validator = Validator::make($request->all(), [
            'addons.*.size' => 'required|string',
            'addons.*.color' => 'required|string',
            'addons.*.featured_image' => 'nullable|image',
        ]);

        $combinations = [];

        foreach ($request->addons as $index => $addon) {
            $size = strtolower(trim($addon['size']));
            $color = strtolower(trim($addon['color']));
            $combo = $size . '|' . $color;

            if (in_array($combo, $combinations)) {
                $validator->after(function ($validator) use ($index) {
                    $validator->errors()->add("addons.$index.size", "Duplicate size-color combination.");
                    $validator->errors()->add("addons.$index.color", "Duplicate size-color combination.");
                });
            }

            $exists = Product::where('sub_sku', $product->sub_sku)
                ->where('size', $addon['size'])
                ->where('color', $addon['color'])
                ->exists();

            if ($exists) {
                $validator->after(function ($validator) use ($index) {
                    $validator->errors()->add("addons.$index.size", "Size-color combo exists.");
                    $validator->errors()->add("addons.$index.color", "Size-color combo exists.");
                });
            }

            $combinations[] = $combo;
        }

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        foreach ($request->addons as $addon) {
            if (isset($addon['featured_image'])) {
                $featured_image = $addon['featured_image'];
                $featuredimageName = time() . '_' . $featured_image->getClientOriginalName();
                $featured_image->move(public_path('uploads/products'), $featuredimageName);
            }

            Product::create([
                'sku' => $product->sku ?? '',
                'hsn_code' => $product->hsn_code ?? '',
                'stock' => $addon['stock'],
                'category_id' => $product->category_id,
                'sub_category_id' => $product->sub_category_id,
                'child_category_id' => $product->child_category_id,
                'brand_id' => $product->brand_id,
                'product_highlights' => $product->product_highlights,
                'product_description' => $product->product_description,
                'length' => $product->length,
                'breadth' => $product->breadth,
                'height' => $product->height,
                'weight' => $product->weight,
                'country_of_origin' => $product->country_of_origin,
                'manufacturer_details' => $product->manufacturer_details,
                'importer_details' => $product->importer_details,
                'packer_details' => $product->packer_details,
                'meta_title' => $product->meta_title,
                'meta_description' => $product->meta_description,
                'meta_keywords' => $product->meta_keywords,
                'tags' => $product->tags,
                'note' => $product->note,
                'status' => $product->status,
                'name' => $product->name,
                'slug' => $product->slug,
                'sub_sku' => $product->sub_sku,
                'product_id' => rand(100000, 999999),
                'size' => $addon['size'],
                'color' => $addon['color'],
                'mrp' => $addon['mrp'],
                'selling_price' => $addon['selling_price'],
                'discount' => $addon['discount'],
                'featured_image' => $featuredimageName ?? '',
                'bulk_images' => $product->bulk_images,
            ]);
        }

        return back()->with('success', 'Variants saved.');
    }
}
